// DOM elements
const closeAllTabsButton = document.getElementById('close-all-tabs');
const tabCountElement = document.getElementById('tab-count');
const autoTimeActionCheckbox = document.getElementById('auto-time-action');
const autoTimeMinutesInput = document.getElementById('auto-time-minutes');
const autoActionTypeSelect = document.getElementById('auto-action-type');
const autoCloseVideosCheckbox = document.getElementById('auto-close-videos');
const closeDelayInput = document.getElementById('close-delay');
const closeIndividualTabToggle = document.getElementById('close-individual-tab-toggle');
const currentTabInfo = document.getElementById('current-tab-info');
const currentTabFavicon = document.getElementById('current-tab-favicon');
const currentTabTitle = document.getElementById('current-tab-title');
const clickBehaviorSelect = document.getElementById('click-behavior-select');
const markTabContainer = document.getElementById('mark-tab-container');
const closePopupButton = document.getElementById('close-popup');
const directionInterfaceEl = document.querySelector('.DirectionInterface');
const leftLane = document.querySelector('.LeftLane');
const rightLane = document.querySelector('.RightLane');
const activeTabPreview = document.querySelector('.ActiveTab');
const triTrack = document.querySelector('.tri-track');

// Track a cleanup timer so OFF state can reset history classes after animations
let directionResetTimer = null;
// Track a cleanup timer for Left/Right swaps so helper classes clear after animation
let directionSwapResetTimer = null;
// Keep current direction state ('left' | 'off' | 'right')
let directionState = 'off';

/**
 * Initialize popup
 * - Attach Debug toggle handler (MV3 blocks inline scripts)
 * - Run other startup tasks with safety guards so Debug always works
 */
document.addEventListener('DOMContentLoaded', async () => {
  // Debug mode removed

  // Temporarily disable toggle animations during init
  try { document.body.classList.add('no-anim'); } catch {}

  // Sync theme to background for toolbar icon
  try {
    const media = window.matchMedia('(prefers-color-scheme: dark)');
    const sendTheme = () => chrome.runtime.sendMessage({ action: 'updateTheme', theme: media.matches ? 'dark' : 'light' });
    sendTheme();
    if (media.addEventListener) {
      media.addEventListener('change', sendTheme);
    } else if (media.addListener) {
      media.addListener(sendTheme);
    }
  } catch (e) {
    console.error('theme sync error:', e);
  }

  // Load settings FIRST so controls reflect saved state before rendering
  try { await loadSettings(); } catch (e) { console.error('loadSettings error:', e); }
  try { await updateTabCount(); } catch (e) { console.error('updateTabCount error:', e); }
  try { await updateCurrentTabInfo(); } catch (e) { console.error('updateCurrentTabInfo error:', e); }
  try { setupEventListeners(); } catch (e) { console.error('setupEventListeners error:', e); }

  // Reveal UI after state is applied to avoid flicker
  try { document.body.classList.remove('preload'); } catch {}

  // Update tab preview after everything is loaded
  try { updateTabPreview(); } catch (e) { console.error('updateTabPreview init error:', e); }

  // Re-enable animations after first paint so user interactions animate normally
  try {
    requestAnimationFrame(() => requestAnimationFrame(() => document.body.classList.remove('no-anim')));
  } catch {}
});

// Event listeners
function setupEventListeners() {
  closeAllTabsButton.addEventListener('click', closeAllYouTubeTabs);
  autoTimeActionCheckbox.addEventListener('change', saveAutoTimeActionSettings);
  autoTimeMinutesInput.addEventListener('change', saveAutoTimeActionSettings);
  autoActionTypeSelect.addEventListener('change', saveAutoTimeActionSettings);
  autoCloseVideosCheckbox.addEventListener('change', saveAutoCloseVideosSettings);
  if (directionInterfaceEl) directionInterfaceEl.addEventListener('click', handleDirectionClick);
  closeDelayInput.addEventListener('change', saveCloseDelaySettings);
  closeIndividualTabToggle.addEventListener('change', handleIndividualTabToggle);
  clickBehaviorSelect.addEventListener('change', saveClickBehavior);
  closePopupButton.addEventListener('click', () => { window.close(); });

  // Keep radios in sync if the hidden select changes (e.g., load/settings)
  clickBehaviorSelect.addEventListener('change', () => {
    const currentValue = clickBehaviorSelect.value;
    const correspondingRadio = document.querySelector(`input[name="click-behavior"][value="${currentValue}"]`);
    if (correspondingRadio && !correspondingRadio.checked) {
      correspondingRadio.checked = true;
    }
  });

  // Wire visible radio group to hidden select (MV3-safe)
  document.querySelectorAll('input[name="click-behavior"]').forEach((radio) => {
    radio.addEventListener('change', function () {
      if (this.checked) {
        clickBehaviorSelect.value = this.value;
        const evt = new Event('change', { bubbles: true });
        clickBehaviorSelect.dispatchEvent(evt);
      }
    });
  });
}

// Update YouTube tab count
async function updateTabCount() {
  try {
    const tabs = await chrome.tabs.query({
      url: ['*://www.youtube.com/*', '*://youtube.com/*', '*://youtu.be/*']
    });
    tabCountElement.textContent = tabs.length;
    
    // Update button text
    closeAllTabsButton.textContent = `Close all YouTube tabs (${tabs.length})`;
    
    // Disable button if no YouTube tabs
    if (tabs.length === 0) {
      closeAllTabsButton.disabled = true;
      closeAllTabsButton.textContent = 'No YouTube tabs';
    } else {
      closeAllTabsButton.disabled = false;
    }
  } catch (error) {
    console.error('Error updating tab count:', error);
    tabCountElement.textContent = '?';
  }
}

// Close all YouTube tabs
async function closeAllYouTubeTabs() {
  try {
    closeAllTabsButton.classList.add('loading');
    closeAllTabsButton.textContent = 'Closing...';
    
    chrome.runtime.sendMessage({
      action: 'closeAllYouTubeTabs'
    });
    
    // Reset button and update count after delay
    setTimeout(async () => {
      closeAllTabsButton.classList.remove('loading');
      await updateTabCount();
    }, 1000);
  } catch (error) {
    console.error('Error closing tabs:', error);
    closeAllTabsButton.classList.remove('loading');
  }
}

/**
 * Update current tab information
 * - Always display current YouTube tab info (favicon + title) if any YouTube tab exists
 * - Prefer active YouTube tab; fallback to most recent YouTube tab
 * - Toggle red icon when marked; show favicon otherwise
 */
async function updateCurrentTabInfo() {
  try {
    const [activeTab] = await chrome.tabs.query({ active: true, currentWindow: true });

    const isYouTube =
      activeTab &&
      activeTab.url &&
      (activeTab.url.includes('youtube.com') || activeTab.url.includes('youtu.be'));

    let selectedTab = null;

    if (isYouTube) {
      selectedTab = activeTab;
    } else {
      const youtubeTabs = await chrome.tabs.query({
        url: ['*://www.youtube.com/*', '*://youtube.com/*', '*://youtu.be/*']
      });
      if (youtubeTabs && youtubeTabs.length > 0) {
        selectedTab = youtubeTabs[0];
      }
    }

    // Ensure info row is visible at all times
    if (currentTabInfo) currentTabInfo.style.display = 'flex';

    const regularIcon = document.getElementById('regular-icon');
    const redIcon = document.getElementById('red-icon');
    // Read autoCloseVideos to determine UI gating for mark toggle
    const { autoCloseVideos: autoCloseEnabled } = await chrome.storage.sync.get(['autoCloseVideos']);

    if (selectedTab) {
      if (currentTabTitle) currentTabTitle.textContent = selectedTab.title || 'YouTube';
      if (currentTabFavicon) {
        if (selectedTab.favIconUrl) {
          currentTabFavicon.src = selectedTab.favIconUrl;
          currentTabFavicon.style.display = 'inline-block';
        } else {
          currentTabFavicon.src = 'icons/Off.png';
          currentTabFavicon.style.display = 'inline-block';
        }
      }

      // Check marked state for the selected tab and toggle icons appropriately
      chrome.runtime.sendMessage(
        { action: 'isTabMarkedForClosing', tabId: selectedTab.id },
        (response) => {
          const isMarked = response && response.isMarked;
          const isWatch = selectedTab.url.includes('youtube.com/watch') || selectedTab.url.includes('youtu.be');

          if (closeIndividualTabToggle) {
            if (autoCloseEnabled) {
              // While auto-close is enabled, mark control is unavailable
              closeIndividualTabToggle.checked = false;
              closeIndividualTabToggle.disabled = true;
            } else {
              // Auto-close is off - allow independent marking on watch pages
              closeIndividualTabToggle.checked = !!isMarked;
              closeIndividualTabToggle.disabled = !isWatch;
            }
          }

          if (autoCloseEnabled) {
            // When auto-close is on, always show YouTube favicon in info row (no red "marked" icon)
            if (redIcon) redIcon.style.display = 'none';
            if (regularIcon) regularIcon.style.display = 'none';
            if (currentTabFavicon) {
              currentTabFavicon.style.display = 'inline-block';
              currentTabFavicon.src = 'https://www.youtube.com/s/desktop/1c6b3028/img/favicon_32x32.png';
            }
          } else if (isMarked) {
            if (redIcon) redIcon.style.display = 'block';
            if (regularIcon) regularIcon.style.display = 'none';
            if (currentTabFavicon) currentTabFavicon.style.display = 'none';
          } else {
            if (redIcon) redIcon.style.display = 'none';
            if (regularIcon) regularIcon.style.display = 'none';
            if (currentTabFavicon) {
              currentTabFavicon.style.display = 'inline-block';
              // Use YouTube favicon when not marked
              currentTabFavicon.src = 'https://www.youtube.com/s/desktop/1c6b3028/img/favicon_32x32.png';
            }
          }
        }
      );
    } else {
      // No YouTube tabs found
      if (currentTabTitle) currentTabTitle.textContent = 'No YouTube tabs found';
      if (currentTabFavicon) currentTabFavicon.style.display = 'none';
      if (redIcon) redIcon.style.display = 'none';
      const regularIconEl = document.getElementById('regular-icon');
      if (regularIconEl) regularIconEl.style.display = 'block';
      if (closeIndividualTabToggle) {
        closeIndividualTabToggle.checked = false;
        closeIndividualTabToggle.disabled = true;
      }
    }
  } catch (error) {
    console.error('Error updating current tab info:', error);
    if (currentTabInfo) currentTabInfo.style.display = 'flex';
  }
}

// Load settings from storage
async function loadSettings() {
  try {
    const result = await chrome.storage.sync.get([
      'autoTimeAction', 
      'autoTimeMinutes',
      'autoActionType',
      'autoCloseVideos', 
      'switchToAdjacentTab',
      'switchDirection',
      'closeDelay',
      'clickBehavior'
    ]);
    
    autoTimeActionCheckbox.checked = result.autoTimeAction || false;
    autoTimeMinutesInput.value = result.autoTimeMinutes || 30;
    autoActionTypeSelect.value = result.autoActionType || 'close';
    autoCloseVideosCheckbox.checked = result.autoCloseVideos || false;

    // Set 3-state direction slider: 0=left, 1=off, 2=right
    const switchToAdjacent = result.switchToAdjacentTab || false;
    const switchDirection = result.switchDirection === 'left' ? 'left' : 'right';
    directionState = switchToAdjacent ? switchDirection : 'off';
    // reflect visuals
    updateTriTrackVisual();
    updateTabPreview();
    
    closeDelayInput.value = (result.closeDelay || 2000) / 1000; // Convert ms to seconds
    
    // Set click behavior select
    const clickBehavior = result.clickBehavior || 'popup';
    clickBehaviorSelect.value = clickBehavior;
    // Reflect stored click behavior to the visible radio group
    const behaviorRadio = document.querySelector(`input[name="click-behavior"][value="${clickBehavior}"]`);
    if (behaviorRadio) behaviorRadio.checked = true;
    
    // Update disabled states
    updateMarkTabState(result.autoCloseVideos || false);
    
  } catch (error) {
    console.error('Error loading settings:', error);
  }
}

// Save auto-close settings
async function saveAutoTimeActionSettings() {
  try {
    const actionType = autoActionTypeSelect.value;
    const isEnabled = autoTimeActionCheckbox.checked;
    
    await chrome.storage.sync.set({
      autoTimeAction: isEnabled,
      autoTimeMinutes: parseInt(autoTimeMinutesInput.value) || 30,
      autoActionType: actionType
    });
    
    // Send message to background script about setting change
    chrome.runtime.sendMessage({
      action: 'updateAutoTimeAction',
      enabled: isEnabled,
      time: parseInt(autoTimeMinutesInput.value) || 30,
      actionType: actionType
    });
  } catch (error) {
    console.error('Error saving auto-time-action settings:', error);
  }
}

// Save auto-close videos settings
async function saveAutoCloseVideosSettings() {
  try {
    await chrome.storage.sync.set({
      autoCloseVideos: autoCloseVideosCheckbox.checked
    });
    
    // Send message to background script about setting change
    chrome.runtime.sendMessage({
      action: 'updateAutoCloseVideos',
      enabled: autoCloseVideosCheckbox.checked
    });
    
    // Update disabled state for mark tab option
    updateMarkTabState(autoCloseVideosCheckbox.checked);
    
    // Re-sync current tab mark state after background finishes unmarking/marking
    setTimeout(() => { try { updateCurrentTabInfo(); } catch (e) {} }, 300);
    
    console.log('Auto-close videos setting saved:', autoCloseVideosCheckbox.checked);
  } catch (error) {
    console.error('Error saving auto-close videos settings:', error);
  }
}

// Save switch to adjacent tab settings
async function saveSwitchToAdjacentSettings() {
  try {
    await chrome.storage.sync.set({
      switchToAdjacentTab: switchToAdjacentCheckbox.checked
    });
    
    // Send message to background script about setting change
    chrome.runtime.sendMessage({
      action: 'updateSwitchToAdjacentTab',
      enabled: switchToAdjacentCheckbox.checked
    });
  } catch (error) {
    console.error('Error saving switch to adjacent tab settings:', error);
  }
}

function normalizeDirectionState(state) {
  if (state === 'left' || state === 'off' || state === 'right') return state;
  if (state === 0 || state === '0') return 'left';
  if (state === 1 || state === '1') return 'off';
  if (state === 2 || state === '2') return 'right';
  return 'off';
}

function handleDirectionClick(event) {
  try {
    const nextState = determineDirectionFromClick(event);
    saveSwitchDirectionSettings(nextState);
  } catch (e) {
    console.error('handleDirectionClick error:', e);
  }
}

function determineDirectionFromClick(event) {
  // If the active tab (or its children) is clicked, treat as Off
  if (activeTabPreview && (event.target === activeTabPreview || activeTabPreview.contains(event.target))) {
    return 'off';
  }

  // Infer by position relative to the interface center so overlaps can't block clicks
  if (directionInterfaceEl) {
    const rect = directionInterfaceEl.getBoundingClientRect();
    const centerX = rect.left + (rect.width / 2);
    return event.clientX <= centerX ? 'left' : 'right';
  }

  return 'off';
}

// Save switch direction settings
async function saveSwitchDirectionSettings(nextState) {
  try {
    const desiredState = normalizeDirectionState(nextState);
    const previousState = directionState;
    directionState = desiredState;

    updateTriTrackVisual();
    updateTabPreview();

    // Avoid redundant writes if the user clicks the same lane repeatedly
    if (previousState === desiredState) return;

    const switchToAdjacent = desiredState !== 'off';
    const direction = desiredState === 'left' ? 'left' : 'right';

    await chrome.storage.sync.set({
      switchToAdjacentTab: switchToAdjacent,
      switchDirection: direction
    });

    // Notify background to apply immediately
    chrome.runtime.sendMessage({
      action: 'updateSwitchToAdjacentTab',
      enabled: switchToAdjacent
    });
    chrome.runtime.sendMessage({
      action: 'updateSwitchDirection',
      direction
    });
  } catch (error) {
    console.error('Error saving switch direction settings:', error);
  }
}

/** Visual state for 3-node track under the direction indicator */
function updateTriTrackVisual() {
  try {
    const track = triTrack || document.querySelector('.tri-track');
    if (!track) return;
    track.classList.remove('is-left', 'is-off', 'is-right');
    if (directionState === 'left') track.classList.add('is-left');
    else if (directionState === 'off') track.classList.add('is-off');
    else track.classList.add('is-right');
  } catch (e) {
    console.error('updateTriTrackVisual error:', e);
  }
}

/** Update tab preview based on direction state */
function updateTabPreview() {
  try {
    const activeTab = activeTabPreview;
    const directionInterface = directionInterfaceEl;
    if (!activeTab || !directionInterface) return;

    // Cancel any pending cleanup from a previous OFF animation
    if (directionResetTimer) {
      clearTimeout(directionResetTimer);
      directionResetTimer = null;
    }
    // Cancel any pending cleanup from a previous L/R animation
    if (directionSwapResetTimer) {
      clearTimeout(directionSwapResetTimer);
      directionSwapResetTimer = null;
    }

    const nextState = normalizeDirectionState(directionState);

    // Determine previous state
    let prevState = 'off';
    if (directionInterface.classList.contains('is-left')) prevState = 'left';
    else if (directionInterface.classList.contains('is-right')) prevState = 'right';

    activeTab.classList.remove('is-left', 'is-off', 'is-right');
    directionInterface.classList.remove('is-left', 'is-off', 'is-right', 'was-off', 'was-left', 'was-right');

    // If we're heading to OFF, strip any L↔R helper classes that zero transitions/lock positions
    const swapHelperClasses = ['from-left', 'from-right', 'lr-prep', 'lr-slide-to-left', 'lr-slide-to-right'];
    if (nextState === 'off') {
      directionInterface.classList.remove(...swapHelperClasses);
      // Force reflow so transition to OFF uses fresh positions (avoids snap after multiple swaps)
      // eslint-disable-next-line no-unused-expressions
      directionInterface.offsetHeight;
    }

    const nextStateClass = `is-${nextState}`;
    activeTab.classList.add(nextStateClass);
    directionInterface.classList.add(nextStateClass);

    // Add history class when the state actually changes; clean it up when the animation finishes
    if (prevState !== nextState) {
      directionInterface.classList.add(`was-${prevState}`);

      // Only clean helper/history when returning to OFF; keep them during Left ↔ Right to preserve smoothness
      if (nextState === 'off') {
        const cleanupClasses = () => {
          directionInterface.classList.remove('was-left', 'was-right', 'was-off', ...swapHelperClasses);
        };

        let cleanupTimeout = null;
        const duration = parseFloat(getComputedStyle(directionInterface).getPropertyValue('--direction-move-duration')) || 0.65;
        const handleCleanup = () => {
          cleanupClasses();
          if (cleanupTimeout) {
            clearTimeout(cleanupTimeout);
            cleanupTimeout = null;
          }
        };

        // Prefer animationend; use a fallback timeout in case animations are disabled
        directionInterface.addEventListener('animationend', handleCleanup, { once: true });
        cleanupTimeout = setTimeout(handleCleanup, (duration * 1000) + 50);
      }
    }
  } catch (e) {
    console.error('updateTabPreview error:', e);
  }
}

// Save close delay settings
async function saveCloseDelaySettings() {
  try {
    const delaySeconds = parseInt(closeDelayInput.value) || 2;
    const delayMs = delaySeconds * 1000; // Convert to milliseconds
    
    await chrome.storage.sync.set({
      closeDelay: delayMs
    });
    
    // Send message to background script about setting change
    chrome.runtime.sendMessage({
      action: 'updateCloseDelay',
      delay: delayMs
    });
    
    console.log('Close delay setting saved:', delayMs + 'ms');
  } catch (error) {
    console.error('Error saving close delay settings:', error);
  }
}

// Handle individual tab toggle
async function handleIndividualTabToggle() {
  try {
    const [currentTab] = await chrome.tabs.query({ active: true, currentWindow: true });
    const isYouTubeWatch =
      currentTab &&
      currentTab.url &&
      (currentTab.url.includes('youtube.com/watch') || currentTab.url.includes('youtu.be'));

    if (isYouTubeWatch) {
      const isChecked = closeIndividualTabToggle.checked;

      // Send message to background to mark/unmark this tab
      chrome.runtime.sendMessage({
        action: 'setTabToCloseAfterVideo',
        tabId: currentTab.id,
        shouldClose: isChecked
      });

      // Refresh UI state (icons/fav)
      await updateCurrentTabInfo();
    } else {
      // Not on YouTube watch page, revert toggle but keep info row visible
      closeIndividualTabToggle.checked = false;
      await updateCurrentTabInfo();
    }
  } catch (error) {
    console.error('Error toggling individual tab:', error);
    closeIndividualTabToggle.checked = false;
    await updateCurrentTabInfo();
  }
}

// Save click behavior preference
async function saveClickBehavior() {
  try {
    const clickBehavior = clickBehaviorSelect.value;
    
    await chrome.storage.sync.set({
      clickBehavior: clickBehavior
    });
    
    // Send message to background script about the change
    chrome.runtime.sendMessage({
      action: 'updateClickBehavior',
      clickBehavior: clickBehavior
    });
    
    console.log('Click behavior saved:', clickBehavior);
  } catch (error) {
    console.error('Error saving click behavior:', error);
  }
}

/**
 * Update mark tab disabled state
 * - Use actual markup (.option-row.mark-tab) instead of missing #mark-tab-container
 */
function updateMarkTabState(autoCloseEnabled) {
  const markTabRow = document.querySelector('.option-row.mark-tab');

  if (autoCloseEnabled) {
    if (markTabRow) markTabRow.classList.add('disabled');
    if (closeIndividualTabToggle) {
      closeIndividualTabToggle.disabled = true;
      closeIndividualTabToggle.checked = false;
    }
    // Keep info row visible; prefer favicon when not marked
    const redIcon = document.getElementById('red-icon');
    const regularIcon = document.getElementById('regular-icon');
    if (redIcon) redIcon.style.display = 'none';
    if (regularIcon) regularIcon.style.display = 'none';
    if (currentTabFavicon) currentTabFavicon.style.display = 'inline-block';
  } else {
    if (markTabRow) markTabRow.classList.remove('disabled');
    if (closeIndividualTabToggle) closeIndividualTabToggle.disabled = false;
    // Re-check current tab state to display info if it's a YouTube tab
    updateCurrentTabInfo();
  }
}
