// Background script (Service Worker) for Close YouTube extension
// This script runs in the background and handles extension lifecycle events

// Lightweight logger and debounce helper
const LOG_PREFIX = '[CloseYouTube][BG]';
let DEBUG_LOGGING = false; // can be toggled later via storage if desired
let CURRENT_THEME = 'light';
function logInfo(...args) { console.log(LOG_PREFIX, ...args); }
function logWarn(...args) { console.warn(LOG_PREFIX, ...args); }
function logError(...args) { console.error(LOG_PREFIX, ...args); }
function logDebug(...args) { if (DEBUG_LOGGING) console.debug(LOG_PREFIX, ...args); }
function debounce(fn, wait) {
  let t = null;
  return (...args) => {
    if (t) clearTimeout(t);
    t = setTimeout(() => fn(...args), wait);
  };
}

logInfo('Background script loaded');

// Extension installation/update handler
chrome.runtime.onInstalled.addListener((details) => {
  logInfo('Extension installed/updated:', details.reason);
  
  if (details.reason === 'install') {
    // Set default settings on first install
    chrome.storage.sync.set({
      autoCloseVideos: false,
      switchToAdjacentTab: false,
      switchDirection: 'right', // default to right
      closeDelay: 2000, // 2 seconds default
      autoClose: false,
      warningTime: 25, // minutes
      autoCloseTime: 30, // minutes
      clickBehavior: 'popup' // default to popup behavior
    });
    
    // Auto-pin the extension if possible
    try {
      chrome.action.setPinned({ pinned: true });
      logInfo('Extension auto-pinned to toolbar');
    } catch (error) {
      logDebug('Could not auto-pin extension:', error);
    }
    
    // Show welcome notification
    chrome.notifications.create({
      type: 'basic',
      iconUrl: 'icons/Off.png',
      title: 'Close YouTube Installed',
      message: 'Pin Close YouTube to your toolbar for recommended access to all features.'
    });
  }
});

// Context menu setup
chrome.runtime.onInstalled.addListener(async () => {
  createContextMenus();
  await initializeState();
});

// Create context menus
async function createContextMenus() {
  // Create context menu for the extension icon
  chrome.contextMenus.create({
    id: 'openOptions',
    title: 'Open options',
    contexts: ['action'] // This makes it appear when right-clicking the extension icon
  });

  // Create context menu for YouTube pages (page context)
  chrome.contextMenus.create({
    id: 'markTabForClosing',
    title: 'Mark tab to close after video',
    contexts: ['all'],
    documentUrlPatterns: ['*://www.youtube.com/*', '*://youtube.com/*', '*://youtu.be/*']
  }, () => {
    if (chrome.runtime.lastError) {
      logError('Error creating context menu:', chrome.runtime.lastError);
    } else {
      logInfo('Context menu created successfully');
    }
  });
  
  // Update context menu based on current state
  await updateContextMenu();
}

// Update context menu based on current state
async function updateContextMenu() {
  try {
    // 'Close tab after video' context menu removed.
    return;
  } catch (error) {
    logError('Error updating context menu:', error);
  }
}

//// Open the action popup anchored to the toolbar icon (not a separate window)
async function openOptionsWindow() {
  try {
    const { clickBehavior = 'popup' } = await chrome.storage.sync.get(['clickBehavior']);
    const wasPopupMode = clickBehavior === 'popup';
    
    // Ensure a popup is configured so we can open it anchored to the icon
    await chrome.action.setPopup({ popup: 'popup.html' });
    
    // Open the anchored popup
    await chrome.action.openPopup();
    
    // If the user's click behavior isn't "popup", restore the no-popup state
    if (!wasPopupMode) {
      await chrome.action.setPopup({ popup: '' });
    }
  } catch (error) {
    logError('Error opening options popup:', error);
  }
}

// Initialize icon state on startup
chrome.runtime.onStartup.addListener(async () => {
  await initializeState();
});

// Update extension icon visibility based on active tab
async function updateIconVisibility() {
  try {
    // Always ensure the extension is enabled
    await chrome.action.enable();
    
    // Get the active tab in the current window
    const [activeTab] = await chrome.tabs.query({
      active: true,
      currentWindow: true
    });
    
    if (activeTab && activeTab.url) {
      const isYouTubeTab = activeTab.url.includes('youtube.com') || activeTab.url.includes('youtu.be');
      
      if (isYouTubeTab) {
        // Get current settings to restore proper icon state
        const result = await chrome.storage.sync.get(['autoCloseVideos']);
        const autoCloseEnabled = result.autoCloseVideos || false;
        
        // Set the correct icon based on auto-close state
        await updateIconState(autoCloseEnabled);
        
        logDebug('Icon shown - active tab is YouTube');
      } else {
        // Use theme-aware icon for non-YouTube tabs
        await updateIconState(false);
        await chrome.action.setTitle({ 
          title: 'Close YouTube - Not on YouTube' 
        });
        await chrome.action.setBadgeText({ text: '' });
        logDebug('Icon dimmed - active tab is not YouTube');
      }
    } else {
      // No active tab, set default icon respecting theme
      await updateIconState(false);
      await chrome.action.setTitle({ title: 'Close YouTube' });
      logDebug('Icon set to default - no active tab');
    }
  } catch (error) {
    logError('Error updating icon visibility:', error);
  }
}

// Handle tab updates to start monitoring on new YouTube tabs
async function refreshDirectionFavicons() {
  try {
    const { switchToAdjacentTab, switchDirection = 'right' } = await chrome.storage.sync.get(['switchToAdjacentTab', 'switchDirection']);
    if (!switchToAdjacentTab) return;
    await applyDirectionFavicons(switchDirection || 'right', switchToAdjacentTab);
  } catch (e) {
    logDebug('refreshDirectionFavicons error', e);
  }
}

const debouncedUpdateIconVisibility = debounce(updateIconVisibility, 120);
const debouncedApplyDirectionFavicons = debounce(refreshDirectionFavicons, 150);

chrome.tabs.onUpdated.addListener(async (tabId, changeInfo, tab) => {
  // Update icon visibility when tab URL changes
  if (changeInfo.status === 'complete' && changeInfo.url) {
    debouncedUpdateIconVisibility();
  }
  
  const isYouTubeTab = tab.url && (tab.url.includes('youtube.com') || tab.url.includes('youtu.be'));
  if (changeInfo.status === 'complete' && isYouTubeTab) {
    // Ensure monitoring starts for watch pages
    try {
      chrome.tabs.sendMessage(tabId, { action: 'startVideoEndMonitoring' }).catch(() => {});
    } catch {}
    // If auto-close is enabled, make favicon behave like marked (but skip directional favicon tabs)
    try {
      const { autoCloseVideos } = await chrome.storage.sync.get(['autoCloseVideos']);
      if (autoCloseVideos && !directionalFaviconTabs.has(tabId)) {
        await ensureOriginalFaviconCached(tabId, tab.favIconUrl);
        await chrome.scripting.executeScript({ target: { tabId }, func: setCustomFavicon });
      }
    } catch {}
    debouncedApplyDirectionFavicons();
  }
});

// Handle tab creation
chrome.tabs.onCreated.addListener(async (tab) => {
  debouncedUpdateIconVisibility();
});

// Handle tab removal
chrome.tabs.onRemoved.addListener(async (tabId) => {
  // Remove from tracking if it was marked for closing
  tabsToCloseAfterVideo.delete(tabId);
  originalFavicons.delete(tabId);
  directionalFaviconTabs.delete(tabId);
  await persistMarkedTabs();
  await persistOriginalFavicons();
  
  // Update icon visibility
  debouncedUpdateIconVisibility();
  debouncedApplyDirectionFavicons();
});

// Inject theme check script into a tab to detect system dark/light mode
async function injectThemeCheck(tabId) {
  try {
    await chrome.scripting.executeScript({
      target: { tabId },
      func: () => {
        const isDark = window.matchMedia('(prefers-color-scheme: dark)').matches;
        chrome.storage.sync.set({ uiTheme: isDark ? 'dark' : 'light' });
      }
    });
  } catch (e) {
    // Tab may not support script injection (chrome:// pages, etc.)
  }
}

// Handle active tab changes (when user switches between tabs)
chrome.tabs.onActivated.addListener(async (activeInfo) => {
  // Check theme on tab switch for responsive toolbar icon updates
  injectThemeCheck(activeInfo.tabId);
  debouncedUpdateIconVisibility();
  debouncedApplyDirectionFavicons();
  // Update context menu for new active tab
  await updateContextMenu();
});

// Handle tab reordering which affects left/right neighbors
chrome.tabs.onMoved.addListener(() => {
  debouncedApplyDirectionFavicons();
});

// Handle window focus changes (when user switches between windows)
chrome.windows.onFocusChanged.addListener(async (windowId) => {
  // Update icon visibility when user switches to a different window
  if (windowId !== chrome.windows.WINDOW_ID_NONE) {
    // Check theme on window focus for responsive toolbar icon updates
    const [activeTab] = await chrome.tabs.query({ active: true, windowId });
    if (activeTab) {
      injectThemeCheck(activeTab.id);
    }
    debouncedUpdateIconVisibility();
    debouncedApplyDirectionFavicons();
  }
});


// Handle extension icon clicks (only when popup is disabled)
chrome.action.onClicked.addListener(async (tab) => {
  logInfo('Extension icon clicked');
  const result = await chrome.storage.sync.get(['clickBehavior']);
  const clickBehavior = result.clickBehavior || 'popup';

  if (clickBehavior === 'popup') {
    // Ensure anchored popup opens even if setPopup hasn't been applied yet.
    await openOptionsWindow();
    return;
  } else if (clickBehavior === 'autoClose') {
    toggleAutoCloseVideos();
  } else if (clickBehavior === 'markTab') {
    if (tab && tab.id && tab.url && (tab.url.includes('youtube.com') || tab.url.includes('youtu.be'))) {
      const isMarked = tabsToCloseAfterVideo.has(tab.id);
      setTabToCloseAfterVideo(tab.id, !isMarked);
    }
  }
});

// Handle context menu clicks
chrome.contextMenus.onClicked.addListener(async (info, tab) => {
  console.log('Context menu item clicked:', info.menuItemId);
  console.log('Tab info:', tab);
  
  if (info.menuItemId === 'openOptions') {
    openOptionsWindow();
  } else if (info.menuItemId === 'markTabForClosingTab') {
    // Right-clicked on a browser tab
    const targetTabId = info.tabId || (tab && tab.id);
    if (targetTabId) {
      const isCurrentlyMarked = tabsToCloseAfterVideo.has(targetTabId);
      setTabToCloseAfterVideo(targetTabId, !isCurrentlyMarked);
    }
  } else if (info.menuItemId === 'markTabForClosing') {
    // Mark the YouTube tab to close after video completion
    // Get the current active tab since we're using 'all' context
    chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
      if (tabs.length > 0) {
        const currentTab = tabs[0];
        if (currentTab.url && (currentTab.url.includes('youtube.com') || currentTab.url.includes('youtu.be'))) {
          setTabToCloseAfterVideo(currentTab.id, true);
        }
      }
    });
  }
});

// Storage for tabs marked to close after video completion
let tabsToCloseAfterVideo = new Set();

// Storage for original favicons
let originalFavicons = new Map();
// Track tabs whose favicons we temporarily swap for direction cues
let directionalFaviconTabs = new Set();

// Session state helpers for MV3 persistence across service worker suspensions
async function loadSessionState() {
  try {
    const { markedTabs = {}, originalFavicons: favs = {} } = await chrome.storage.session.get(['markedTabs', 'originalFavicons']);
    // Restore marked tabs
    tabsToCloseAfterVideo = new Set(
      Object.keys(markedTabs)
        .filter((k) => !!markedTabs[k])
        .map((k) => Number(k))
    );
    // Restore original favicons
    originalFavicons = new Map(
      Object.keys(favs).map((k) => [Number(k), favs[k]])
    );
    // Directional swaps are ephemeral; reset each session
    directionalFaviconTabs = new Set();
    logDebug('Session state loaded', { markedCount: tabsToCloseAfterVideo.size, favCount: originalFavicons.size });
  } catch (e) {
    logError('Failed to load session state', e);
  }
}

// Force-close a tab due to time limit, regardless of mark state
async function forceCloseTab(tabId) {
  try {
    const settings = await chrome.storage.sync.get(['switchToAdjacentTab', 'closeDelay']);
    if (settings.switchToAdjacentTab) {
      await switchToAdjacentYouTubeTab(tabId);
    }
    const delay = settings.closeDelay || 2000;
    setTimeout(async () => {
      try {
        await chrome.tabs.remove(tabId);
      } catch (e) {
        console.log('Tab already closed or cannot close:', tabId, e);
      }
      // Clean up any mark state
      if (tabsToCloseAfterVideo.delete(tabId)) {
        await persistMarkedTabs();
      }
    }, delay);
  } catch (e) {
    console.error('Error force-closing tab:', tabId, e);
  }
}

async function persistMarkedTabs() {
  try {
    const obj = {};
    for (const id of tabsToCloseAfterVideo) obj[id] = true;
    await chrome.storage.session.set({ markedTabs: obj });
  } catch (e) {
    logError('Failed to persist marked tabs', e);
  }
}

async function persistOriginalFavicons() {
  try {
    const obj = {};
    for (const [id, url] of originalFavicons.entries()) obj[id] = url || '';
    await chrome.storage.session.set({ originalFavicons: obj });
  } catch (e) {
    logError('Failed to persist original favicons', e);
  }
}

/**
 * Try to read the page's favicon link from the DOM (absolute URL).
 */
async function getPageFavicon(tabId) {
  try {
    const results = await chrome.scripting.executeScript({
      target: { tabId },
      func: () => {
        try {
          const toAbs = (href) => {
            try { return new URL(href, document.baseURI).href; } catch { return href; }
          };
          const links = Array.from(document.querySelectorAll("link[rel*='icon']"));
          if (links.length === 0) return null;
          const preferred = links.find(l => l.sizes && /(?:32|48|64|96|128)/.test(l.sizes.value)) || links[0];
          return toAbs(preferred.href);
        } catch {
          return null;
        }
      }
    });
    if (results && results[0] && typeof results[0].result === 'string') {
      return results[0].result;
    }
    return null;
  } catch (e) {
    logDebug('getPageFavicon failed', e);
    return null;
  }
}

/**
 * Ensure we have a cached original favicon for a tab, using the Tabs API value
 * and falling back to reading from the page DOM if needed.
 */
async function getBestFaviconUrl(tabId) {
  try {
    const tab = await chrome.tabs.get(tabId);
    // Prefer DOM-fetched icon because it reflects the latest state after our restore
    const fromDom = await getPageFavicon(tabId);
    return fromDom || (tab && tab.favIconUrl) || null;
  } catch (e) {
    return null;
  }
}

async function ensureOriginalFaviconCached(tabId, fallbackFavIconUrl = null) {
  try {
    if (!originalFavicons.has(tabId) || !originalFavicons.get(tabId)) {
      let original = fallbackFavIconUrl || null;
      if (!original) {
        original = await getPageFavicon(tabId);
      }
      originalFavicons.set(tabId, original || '');
      await persistOriginalFavicons();
      logDebug('Cached original favicon for tab', tabId, original);
    }
  } catch (e) {
    logDebug('ensureOriginalFaviconCached error', e);
  }
}

async function reconcileMarkedTabs() {
  try {
    const { autoCloseVideos, switchToAdjacentTab } = await chrome.storage.sync.get(['autoCloseVideos', 'switchToAdjacentTab']);
    const youTubeTabs = await chrome.tabs.query({
      url: ['*://www.youtube.com/*', '*://youtube.com/*', '*://youtu.be/*']
    });

    // Find active YouTube tab for direction mode
    const activeYouTubeTab = youTubeTabs.find(t => t.active);

    for (const tab of youTubeTabs) {
      // Skip tabs with directional favicons - they override marked/auto-close favicons
      if (directionalFaviconTabs.has(tab.id)) {
        continue;
      }

      // When direction mode is active, only apply red X to active tab
      if (switchToAdjacentTab && activeYouTubeTab && tab.id !== activeYouTubeTab.id) {
        continue;
      }
      
      const isWatch = tab.url && (tab.url.includes('youtube.com/watch') || tab.url.includes('youtu.be/'));
      const shouldShowCustom = tabsToCloseAfterVideo.has(tab.id) || (autoCloseVideos && isWatch);

      if (shouldShowCustom) {
        // Ensure monitoring is active for watch tabs
        if (isWatch) {
          chrome.tabs.sendMessage(tab.id, { action: 'startVideoEndMonitoring' }).catch(err => logDebug('Could not send start monitor during reconcile:', err));
        }
        // Ensure original favicon stored
        if (!originalFavicons.has(tab.id)) {
          originalFavicons.set(tab.id, tab.favIconUrl);
        }
        // Set custom favicon
        try {
          await chrome.scripting.executeScript({ target: { tabId: tab.id }, func: setCustomFavicon });
        } catch (e) {
          logDebug('Failed to set custom favicon during reconcile', e);
        }
      } else {
        const original = originalFavicons.get(tab.id) || null;
        try {
          await chrome.scripting.executeScript({
            target: { tabId: tab.id },
            func: restoreOriginalFavicon,
            args: [original]
          });
        } catch (e) {
          logDebug('Failed to restore favicon during reconcile', e);
        }
        originalFavicons.delete(tab.id);
      }
    }
    await persistOriginalFavicons();
    logDebug('Reconcile complete');
  } catch (e) {
    logError('Error during reconcileMarkedTabs', e);
  }
}

// If global autoCloseVideos is enabled, ensure existing watch tabs are marked and monitored
async function applyAutoCloseIfEnabled() {
  try {
    const { autoCloseVideos } = await chrome.storage.sync.get(['autoCloseVideos']);
    if (!autoCloseVideos) return;

    // Start monitoring on existing watch tabs, and set favicons like "marked"
    const youTubeWatchTabs = await chrome.tabs.query({
      url: ['*://www.youtube.com/watch*', '*://youtube.com/watch*', '*://youtu.be/*']
    });

    for (const tab of youTubeWatchTabs) {
      try {
        chrome.tabs.sendMessage(tab.id, { action: 'startVideoEndMonitoring' }).catch(() => {});
      } catch {}
      try {
        // Skip tabs with directional favicons - they override auto-close favicons
        if (directionalFaviconTabs.has(tab.id)) {
          continue;
        }
        await ensureOriginalFaviconCached(tab.id, tab.favIconUrl);
        await chrome.scripting.executeScript({ target: { tabId: tab.id }, func: setCustomFavicon });
      } catch {}
    }
  } catch (e) {
    logDebug('applyAutoCloseIfEnabled error', e);
  }
}

async function initializeState() {
  await loadSessionState();
  await initializeIconState();
  await applyAutoCloseIfEnabled();
  await reconcileMarkedTabs();
  await updateIconVisibility();
  await updateContextMenu();
  try {
    const { switchDirection = 'right', switchToAdjacentTab = false } = await chrome.storage.sync.get(['switchDirection', 'switchToAdjacentTab']);
    await applyDirectionFavicons(switchDirection || 'right', switchToAdjacentTab);
  } catch (e) {
    logDebug('initializeState direction favicons error', e);
  }
}

// Functions to be executed in page context for favicon changes
function setCustomFavicon(iconOverride) {
  try {
    const doc = document;
    const base = iconOverride || chrome.runtime.getURL('icons/Favicon.png');
    const iconUrl = `${base}${base.includes('?') ? '&' : '?'}cy=${Date.now()}`;

    // Strategy A: Overwrite existing icon links and cache originals for restore
    const existing = Array.from(doc.querySelectorAll('link[rel*="icon"]'));
    if (existing.length > 0) {
      existing.forEach((l) => {
        try {
          if (!l.dataset.closeYoutubeOrig) {
            l.dataset.closeYoutubeOrig = l.href || '';
          }
          l.type = 'image/png';
          l.href = iconUrl;
        } catch {}
      });
      return;
    }

    // Strategy B: No icons present - inject our PNG (and shortcut icon) with a quick refresh
    // Remove any old injected icons from previous states
    Array.from(doc.querySelectorAll('[id^="close-youtube-custom-favicon"]')).forEach((n) => n.remove());

    // Small refresh hop helps Chrome pick up changes
    const blank = 'data:image/gif;base64,R0lGODlhAQABAAD/ACwAAAAAAQABAAACADs=';
    const tmp1 = doc.createElement('link');
    tmp1.id = 'close-youtube-temp-favicon';
    tmp1.rel = 'icon';
    tmp1.href = blank;
    doc.head.appendChild(tmp1);

    const tmp2 = doc.createElement('link');
    tmp2.id = 'close-youtube-temp-favicon-alt';
    tmp2.rel = 'shortcut icon';
    tmp2.href = blank;
    doc.head.appendChild(tmp2);

    setTimeout(() => {
      try {
        const t1 = doc.getElementById('close-youtube-temp-favicon');
        if (t1) t1.remove();
        const t2 = doc.getElementById('close-youtube-temp-favicon-alt');
        if (t2) t2.remove();

        const l1 = doc.createElement('link');
        l1.id = 'close-youtube-custom-favicon';
        l1.rel = 'icon';
        l1.type = 'image/png';
        l1.sizes = '16x16';
        l1.href = iconUrl;
        doc.head.appendChild(l1);

        const l2 = doc.createElement('link');
        l2.id = 'close-youtube-custom-favicon-alt';
        l2.rel = 'shortcut icon';
        l2.type = 'image/png';
        l2.sizes = '16x16';
        l2.href = iconUrl;
        doc.head.appendChild(l2);
      } catch (e) {
        console.error('Error injecting custom favicon PNG:', e);
      }
    }, 50);
  } catch (error) {
    console.error('Error setting custom favicon:', error);
  }
}

// Set horizontally flipped YouTube icon as favicon
function setFlippedYouTubeFavicon(youtubeIconUrl) {
  try {
    const doc = document;
    
    // Create an image to load the YouTube icon
    const img = new Image();
    
    img.onload = function() {
      try {
        // Create canvas and flip the image horizontally
        const canvas = doc.createElement('canvas');
        canvas.width = img.width;
        canvas.height = img.height;
        const ctx = canvas.getContext('2d');
        
        // Flip horizontally by translating and scaling
        ctx.translate(canvas.width, 0);
        ctx.scale(-1, 1);
        ctx.drawImage(img, 0, 0);
        
        // Convert to data URL
        const flippedDataUrl = canvas.toDataURL('image/png');
        const iconUrl = `${flippedDataUrl}${flippedDataUrl.includes('?') ? '&' : '?'}cy=${Date.now()}`;
        
        // Strategy A: Overwrite existing icon links and cache originals for restore
        const existing = Array.from(doc.querySelectorAll('link[rel*="icon"]'));
        if (existing.length > 0) {
          existing.forEach((l) => {
            try {
              if (!l.dataset.closeYoutubeOrig) {
                l.dataset.closeYoutubeOrig = l.href || '';
              }
              l.type = 'image/png';
              l.href = iconUrl;
            } catch {}
          });
          return;
        }
        
        // Strategy B: No icons present - inject flipped icon
        Array.from(doc.querySelectorAll('[id^="close-youtube-custom-favicon"]')).forEach((n) => n.remove());
        
        const blank = 'data:image/gif;base64,R0lGODlhAQABAAD/ACwAAAAAAQABAAACADs=';
        const tmp1 = doc.createElement('link');
        tmp1.id = 'close-youtube-temp-favicon';
        tmp1.rel = 'icon';
        tmp1.href = blank;
        doc.head.appendChild(tmp1);
        
        const tmp2 = doc.createElement('link');
        tmp2.id = 'close-youtube-temp-favicon-alt';
        tmp2.rel = 'shortcut icon';
        tmp2.href = blank;
        doc.head.appendChild(tmp2);
        
        setTimeout(() => {
          try {
            const t1 = doc.getElementById('close-youtube-temp-favicon');
            if (t1) t1.remove();
            const t2 = doc.getElementById('close-youtube-temp-favicon-alt');
            if (t2) t2.remove();
            
            const l1 = doc.createElement('link');
            l1.id = 'close-youtube-custom-favicon';
            l1.rel = 'icon';
            l1.type = 'image/png';
            l1.sizes = '16x16';
            l1.href = iconUrl;
            doc.head.appendChild(l1);
            
            const l2 = doc.createElement('link');
            l2.id = 'close-youtube-custom-favicon-alt';
            l2.rel = 'shortcut icon';
            l2.type = 'image/png';
            l2.sizes = '16x16';
            l2.href = iconUrl;
            doc.head.appendChild(l2);
          } catch (e) {
            console.error('Error injecting flipped YouTube favicon:', e);
          }
        }, 50);
      } catch (e) {
        console.error('Error creating flipped favicon:', e);
      }
    };
    
    img.onerror = function() {
      console.error('Failed to load YouTube icon for flipping');
    };
    
    img.src = youtubeIconUrl;
  } catch (error) {
    console.error('Error setting flipped YouTube favicon:', error);
  }
}

function restoreOriginalFavicon(originalFavicon) {
  try {
    const doc = document;

    // STRATEGY A: If we overwrote existing links, restore them from dataset
    let restored = false;
    Array.from(doc.querySelectorAll('link[rel*="icon"]')).forEach((l) => {
      const orig = l.dataset && l.dataset.closeYoutubeOrig;
      if (typeof orig !== 'undefined') {
        try {
          // Direct restore - no blank/delay to avoid grey globe flicker
          l.href = orig || l.href;
          delete l.dataset.closeYoutubeOrig;
          restored = true;
        } catch {}
      }
    });
    if (restored) {
      console.log('Original favicon restored by reverting existing icon links');
      return;
    }

    // STRATEGY B: Remove all our injected variants (16/32 + alts)
    Array.from(doc.querySelectorAll('[id^="close-youtube-custom-favicon"]')).forEach(n => n.remove());
    // Fallback: remove any extension-injected icon links (older versions may have added without ids)
    try {
      doc.querySelectorAll('link[rel*="icon"]').forEach(l => {
        const href = l.getAttribute('href') || '';
        if (href.startsWith('chrome-extension://')) {
          l.remove();
        }
      });
    } catch {}

    // Force refresh pass: blank first, then set original (if known) or rely on site's own
    const blank = 'data:image/gif;base64,R0lGODlhAQABAAD/ACwAAAAAAQABAAACADs=';

    // Insert blank temp to break caching
    const tmp = doc.createElement('link');
    tmp.id = 'close-youtube-temp-favicon';
    tmp.rel = 'icon';
    tmp.href = blank;
    doc.head.appendChild(tmp);

    const tmpAlt = doc.createElement('link');
    tmpAlt.id = 'close-youtube-temp-favicon-alt';
    tmpAlt.rel = 'shortcut icon';
    tmpAlt.href = blank;
    doc.head.appendChild(tmpAlt);

    setTimeout(() => {
      try {
        const t = doc.getElementById('close-youtube-temp-favicon');
        if (t) t.remove();
        const ta = doc.getElementById('close-youtube-temp-favicon-alt');
        if (ta) ta.remove();

        if (originalFavicon) {
          // Cache-bust to force update
          const cacheBusted = (() => {
            try {
              const u = new URL(originalFavicon, doc.baseURI);
              u.searchParams.set('cy', Date.now().toString());
              return u.href;
            } catch {
              return originalFavicon;
            }
          })();

          const link = doc.createElement('link');
          link.id = 'close-youtube-restore-favicon';
          link.rel = 'icon';
          link.href = cacheBusted;
          doc.head.appendChild(link);

          const linkAlt = doc.createElement('link');
          linkAlt.id = 'close-youtube-restore-favicon-alt';
          linkAlt.rel = 'shortcut icon';
          linkAlt.href = cacheBusted;
          doc.head.appendChild(linkAlt);
        }
      } catch (e) {
        console.error('Error swapping back to original favicon:', e);
      }
    }, 50);

    console.log('Original favicon restore requested (with refresh):', originalFavicon || '(removed custom only)');
  } catch (error) {
    console.error('Error restoring original favicon:', error);
  }
}

// Handle messages from popup and content scripts
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  console.log('Background received message:', message);
  
  switch (message.action) {
    case 'videoEnded':
      handleVideoEnded(sender.tab.id);
      break;
      
    case 'updateAutoCloseVideos':
      handleAutoCloseVideosUpdate(message.enabled);
      break;
      
    case 'updateSwitchToAdjacentTab':
      handleSwitchToAdjacentTabUpdate(message.enabled);
      break;
      
    case 'updateSwitchDirection':
      handleSwitchDirectionUpdate(message.direction);
      break;
      
    case 'updateCloseDelay':
      handleCloseDelayUpdate(message.delay);
      break;
      
    case 'updateAutoTimeAction':
      handleAutoTimeActionUpdate(message.enabled, message.time, message.actionType);
      break;
      
    case 'closeAllYouTubeTabs':
      closeAllYouTubeTabs();
      break;
      
    case 'getYouTubeTabCount':
      getYouTubeTabCount().then(count => sendResponse({ count }));
      return true; // Keep message channel open for async response
      
    case 'setTabToCloseAfterVideo':
      setTabToCloseAfterVideo(message.tabId, message.shouldClose);
      break;
      
    case 'updateClickBehavior':
      handleClickBehaviorUpdate(message.clickBehavior);
      break;
      
    case 'isTabMarkedForClosing':
      sendResponse({ isMarked: tabsToCloseAfterVideo.has(message.tabId) });
      return true;

    case 'updateTheme':
      (async () => {
        try {
          CURRENT_THEME = (message.theme === 'dark') ? 'dark' : 'light';
          await chrome.storage.sync.set({ uiTheme: CURRENT_THEME });
          const { autoCloseVideos } = await chrome.storage.sync.get(['autoCloseVideos']);
          await updateIconState(!!autoCloseVideos);
        } catch (e) {
          logDebug('Failed to update theme', e);
        }
      })();
      break;

    case 'getTabFavicon':
      (async () => {
        try {
          const url = await getBestFaviconUrl(message.tabId);
          sendResponse({ url: url || null });
        } catch (e) {
          sendResponse({ url: null });
        }
      })();
      return true;
      
    default:
      console.log('Unknown message action:', message.action);
  }
});


// Handle video ended event
async function handleVideoEnded(tabId) {
  console.log(`Video ended in tab ${tabId}`);
  
  const settings = await chrome.storage.sync.get(['autoCloseVideos', 'switchToAdjacentTab', 'closeDelay']);
  const shouldCloseSpecificTab = tabsToCloseAfterVideo.has(tabId);
  const shouldAutoCloseAll = settings.autoCloseVideos;
  
  if (shouldCloseSpecificTab || shouldAutoCloseAll) {
    console.log(`Auto-closing tab ${tabId} after video ended`);
    
    // Handle tab switching before closing
    if (settings.switchToAdjacentTab) {
      await switchToAdjacentYouTubeTab(tabId);
    }

    // Close immediately (no delay)
    try {
      await chrome.tabs.remove(tabId);
    } catch (err) {
      console.log('Tab already closed:', err);
    }
    if (tabsToCloseAfterVideo.delete(tabId)) {
      await persistMarkedTabs();
    }
  }
}


// Set tab to close after video completion
async function setTabToCloseAfterVideo(tabId, shouldClose) {
  // Drop any directional icon when the tab is explicitly marked/unmarked
  directionalFaviconTabs.delete(tabId);

  if (shouldClose) {
    tabsToCloseAfterVideo.add(tabId);
    await persistMarkedTabs();
    console.log(`Tab ${tabId} marked to close after video ends`);
    
    // Send message to content script to start monitoring
    chrome.tabs.sendMessage(tabId, {
      action: 'startVideoEndMonitoring'
    }).catch(err => console.log('Could not send message to tab:', err));
    
    // Change the tab's favicon to show it's marked for closing
    await changeTabFavicon(tabId, true);
    
    // Check if all YouTube tabs are now marked for closing
    await checkAllTabsMarked();

    // Update context menu
    await updateContextMenu();
  } else {
    tabsToCloseAfterVideo.delete(tabId);
    await persistMarkedTabs();
    console.log(`Tab ${tabId} unmarked from closing after video ends`);
    
    // Send message to content script to stop monitoring
    chrome.tabs.sendMessage(tabId, {
      action: 'stopVideoEndMonitoring'
    }).catch(err => console.log('Could not send message to tab:', err));
    
    // Restore the original favicon
    await changeTabFavicon(tabId, false);
    
    // Check if we need to restore other tabs' favicons
    await checkAllTabsMarked();

    // Update context menu
    await updateContextMenu();
  }
}

// Change tab favicon to show closing status
async function changeTabFavicon(tabId, isMarkedForClosing) {
  try {
    const tab = await chrome.tabs.get(tabId);
    console.log(`Changing favicon for tab ${tabId}, marked: ${isMarkedForClosing}`);
    
    if (isMarkedForClosing) {
      // Ensure original favicon is cached (from Tabs API or page DOM)
      await ensureOriginalFaviconCached(tabId, tab.favIconUrl);
      
      // Set the custom favicon
      console.log(`Setting custom favicon for tab ${tabId}`);
      await chrome.scripting.executeScript({
        target: { tabId: tabId },
        func: setCustomFavicon
      });
      console.log(`Custom favicon set for tab ${tabId}`);
    } else {
      const originalFavicon = originalFavicons.get(tabId) || null;
      console.log(`Restoring favicon for tab ${tabId}:`, originalFavicon || '(remove custom)');
      await chrome.scripting.executeScript({
        target: { tabId: tabId },
        func: restoreOriginalFavicon,
        args: [originalFavicon]
      });
      if (originalFavicons.has(tabId)) {
        originalFavicons.delete(tabId);
        await persistOriginalFavicons();
      }
      console.log(`Favicon restored/cleaned for tab ${tabId}`);
    }
  } catch (error) {
    console.error('Could not change favicon for tab:', tabId, error);
  }
}

// Check if all YouTube tabs are marked for closing
async function checkAllTabsMarked() {
  try {
    const youTubeTabs = await chrome.tabs.query({
      url: ['*://www.youtube.com/*', '*://youtube.com/*', '*://youtu.be/*']
    });
    
    const allMarked = youTubeTabs.every(tab => tabsToCloseAfterVideo.has(tab.id));
    
    if (allMarked && youTubeTabs.length > 0) {
      // All YouTube tabs are marked, change all favicons
      for (const tab of youTubeTabs) {
        if (!originalFavicons.has(tab.id)) {
          originalFavicons.set(tab.id, tab.favIconUrl);
        }
        await chrome.scripting.executeScript({
          target: { tabId: tab.id },
          func: setCustomFavicon
        });
      }
    } else {
      // Not all tabs are marked, ensure unmarked tabs use their natural favicon
      for (const tab of youTubeTabs) {
        if (!tabsToCloseAfterVideo.has(tab.id)) {
          const originalFavicon = originalFavicons.get(tab.id) || null;
          await chrome.scripting.executeScript({
            target: { tabId: tab.id },
            func: restoreOriginalFavicon,
            args: [originalFavicon]
          });
          originalFavicons.delete(tab.id);
        }
      }
    }
  } catch (error) {
    console.log('Error checking all tabs marked status:', error);
  }
}

async function restoreDirectionalFavicons() {
  const restoreTargets = Array.from(directionalFaviconTabs);
  const { autoCloseVideos } = await chrome.storage.sync.get(['autoCloseVideos']);

  // Restore all tabs in parallel for faster response
  await Promise.all(restoreTargets.map(async (tabId) => {
    // Skip manually marked tabs (they keep their mark icon)
    if (tabsToCloseAfterVideo.has(tabId)) {
      directionalFaviconTabs.delete(tabId);
      return;
    }
    try {
      if (autoCloseVideos) {
        // Auto-close is on - apply custom red favicon
        await chrome.scripting.executeScript({
          target: { tabId },
          func: setCustomFavicon
        });
      } else {
        // Restore to original
        const original = originalFavicons.get(tabId) || null;
        await chrome.scripting.executeScript({
          target: { tabId },
          func: restoreOriginalFavicon,
          args: [original]
        });
      }
    } catch (e) {
      logDebug('Failed to restore directional favicon', e);
    } finally {
      directionalFaviconTabs.delete(tabId);
    }
  }));
}

async function applyDirectionFavicons(direction, enabledOverride = null) {
  try {
    const { switchToAdjacentTab, autoCloseVideos } = await chrome.storage.sync.get(['switchToAdjacentTab', 'autoCloseVideos']);
    const enabled = typeof enabledOverride === 'boolean' ? enabledOverride : switchToAdjacentTab;
    if (!enabled) {
      await restoreDirectionalFavicons();
      return;
    }

    const allTabs = await chrome.tabs.query({ currentWindow: true });
    const youTubeTabs = allTabs.filter(
      (tab) => tab.url && (tab.url.includes('youtube.com') || tab.url.includes('youtu.be'))
    );

    if (youTubeTabs.length === 0) {
      await restoreDirectionalFavicons();
      return;
    }

    const referenceTab =
      youTubeTabs.find((t) => t.active) ||
      youTubeTabs.reduce((latest, tab) => {
        if (!latest) return tab;
        return (tab.lastAccessed || 0) > (latest.lastAccessed || 0) ? tab : latest;
      }, null);

    if (!referenceTab) {
      await restoreDirectionalFavicons();
      return;
    }

    const targetIds = new Set();

    for (const tab of youTubeTabs) {
      // Skip tabs that are explicitly marked (they keep their mark icon)
      if (tabsToCloseAfterVideo.has(tab.id)) continue;

      const isTarget = direction === 'left' ? tab.index < referenceTab.index : tab.index > referenceTab.index;
      if (!isTarget) continue;

      targetIds.add(tab.id);
      try {
        await ensureOriginalFaviconCached(tab.id, tab.favIconUrl);
        
        if (direction === 'left') {
          // Use pre-flipped YouTube icon for left direction
          const iconUrl = chrome.runtime.getURL('icons/YTiconLeft.png');
          await chrome.scripting.executeScript({
            target: { tabId: tab.id },
            func: setCustomFavicon,
            args: [iconUrl]
          });
          directionalFaviconTabs.add(tab.id);
        } else if (direction === 'right') {
          // Use YouTube icon with right arrow for right direction
          const iconUrl = chrome.runtime.getURL('icons/YTicon.png');
          await chrome.scripting.executeScript({
            target: { tabId: tab.id },
            func: setCustomFavicon,
            args: [iconUrl]
          });
          directionalFaviconTabs.add(tab.id);
        }
      } catch (e) {
        logDebug('Failed to set directional favicon', e);
      }
    }

    // Restore any tabs we previously modified that are no longer targets (cleanup old state first)
    const toRestore = Array.from(directionalFaviconTabs).filter((id) => !targetIds.has(id));
    for (const tabId of toRestore) {
      if (tabsToCloseAfterVideo.has(tabId)) {
        directionalFaviconTabs.delete(tabId);
        continue;
      }
      try {
        const original = originalFavicons.get(tabId) || null;
        await chrome.scripting.executeScript({
          target: { tabId },
          func: restoreOriginalFavicon,
          args: [original]
        });
      } catch (e) {
        logDebug('Failed to restore outdated directional favicon', e);
      } finally {
        directionalFaviconTabs.delete(tabId);
      }
    }

    // Restore original favicon for tabs on the non-target side (except active tab)
    for (const tab of youTubeTabs) {
      // Skip the active/reference tab (it keeps the red X)
      if (tab.id === referenceTab.id) continue;
      // Skip manually marked tabs
      if (tabsToCloseAfterVideo.has(tab.id)) continue;
      // Skip target-side tabs (they have direction icons)
      if (targetIds.has(tab.id)) continue;

      // Add to directionalFaviconTabs so other code won't re-apply red X
      directionalFaviconTabs.add(tab.id);

      // This tab is on the non-target side - restore original favicon
      try {
        await ensureOriginalFaviconCached(tab.id, tab.favIconUrl);
        const original = originalFavicons.get(tab.id) || null;
        await chrome.scripting.executeScript({
          target: { tabId: tab.id },
          func: restoreOriginalFavicon,
          args: [original]
        });
      } catch (e) {
        logDebug('Failed to restore non-target side favicon', e);
      }
    }
  } catch (e) {
    logDebug('applyDirectionFavicons error', e);
  }
}

// Switch to adjacent YouTube tab
async function switchToAdjacentYouTubeTab(currentTabId) {
  try {
    const allTabs = await chrome.tabs.query({ currentWindow: true });
    const youTubeTabs = allTabs.filter(tab => 
      tab.url.includes('youtube.com/watch') && tab.id !== currentTabId
    );
    
    if (youTubeTabs.length === 0) return;
    
    // Find current tab index
    const currentTab = allTabs.find(tab => tab.id === currentTabId);
    if (!currentTab) return;
    
    const currentIndex = allTabs.indexOf(currentTab);
    
    // Get user's preferred direction
    const settings = await chrome.storage.sync.get(['switchDirection']);
    const preferredDirection = settings.switchDirection || 'right';
    
    let targetTab = null;
    
    if (preferredDirection === 'left') {
      // Look for YouTube tab to the left first
      for (let i = currentIndex - 1; i >= 0; i--) {
        if (allTabs[i].url.includes('youtube.com/watch')) {
          targetTab = allTabs[i];
          break;
        }
      }
      
      // If not found to the left, look to the right
      if (!targetTab) {
        for (let i = currentIndex + 1; i < allTabs.length; i++) {
          if (allTabs[i].url.includes('youtube.com/watch')) {
            targetTab = allTabs[i];
            break;
          }
        }
      }
    } else {
      // Look for YouTube tab to the right first
      for (let i = currentIndex + 1; i < allTabs.length; i++) {
        if (allTabs[i].url.includes('youtube.com/watch')) {
          targetTab = allTabs[i];
          break;
        }
      }
      
      // If not found to the right, look to the left
      if (!targetTab) {
        for (let i = currentIndex - 1; i >= 0; i--) {
          if (allTabs[i].url.includes('youtube.com/watch')) {
            targetTab = allTabs[i];
            break;
          }
        }
      }
    }
    
    // If still not found, use any YouTube tab
    if (!targetTab && youTubeTabs.length > 0) {
      targetTab = youTubeTabs[0];
    }
    
    if (targetTab) {
      await chrome.tabs.update(targetTab.id, { active: true });
      try {
        chrome.tabs.sendMessage(targetTab.id, { action: 'playIfPaused' }).catch(() => {});
      } catch {}
      console.log(`Switched to adjacent YouTube tab: ${targetTab.id}`);
    }
  } catch (error) {
    console.error('Error switching to adjacent tab:', error);
  }
}



/** Update extension icon based on auto-close state and theme */
async function updateIconState(enabled) {
  const isDark = (CURRENT_THEME === 'dark');

  const iconPath = enabled
    ? (isDark
        ? {
            "16": "icons/DarkClose YouTube Activated.png",
            "32": "icons/DarkClose YouTube Activated.png",
            "48": "icons/DarkClose YouTube Activated.png",
            "128": "icons/DarkClose YouTube Activated.png"
          }
        : {
            "16": "icons/On.png",
            "32": "icons/On.png",
            "48": "icons/On.png",
            "128": "icons/On.png"
          })
    : (isDark
        ? {
            "16": "icons/DarkClose YouTube Off.png",
            "32": "icons/DarkClose YouTube Off.png",
            "48": "icons/DarkClose YouTube Off.png",
            "128": "icons/DarkClose YouTube Off.png"
          }
        : {
            "16": "icons/Off.png",
            "32": "icons/Off.png",
            "48": "icons/Off.png",
            "128": "icons/Off.png"
          });

  const title = enabled
    ? 'Close YouTube - Auto-close ENABLED'
    : 'Close YouTube - Auto-close DISABLED';

  await chrome.action.setIcon({ path: iconPath });
  await chrome.action.setTitle({ title });
  await chrome.action.setBadgeText({ text: '' });
}

// Initialize icon state and popup behavior on startup
async function initializeIconState() {
  const result = await chrome.storage.sync.get(['autoCloseVideos', 'clickBehavior', 'uiTheme']);
  const enabled = result.autoCloseVideos || false;
  const clickBehavior = result.clickBehavior || 'popup';
  CURRENT_THEME = result.uiTheme || CURRENT_THEME;

  await updateIconState(enabled);

  // Set popup behavior based on user preference
  if (clickBehavior === 'popup') {
    await chrome.action.setPopup({ popup: 'popup.html' });
  } else {
    await chrome.action.setPopup({ popup: '' });
  }
}

// Toggle auto-close videos setting
async function toggleAutoCloseVideos() {
  const result = await chrome.storage.sync.get(['autoCloseVideos']);
  const currentState = result.autoCloseVideos || false;
  const newState = !currentState;
  
  await chrome.storage.sync.set({ autoCloseVideos: newState });
  
  // Update icon state
  await updateIconState(newState);

  // Update context menu
  await updateContextMenu();

  // Apply side effects (start/stop monitors) without altering individual marks
  handleAutoCloseVideosUpdate(newState);
  
  console.log('Auto-close videos toggled to:', newState);
}

// Close all YouTube tabs
async function closeAllYouTubeTabs() {
  try {
    const youTubeTabs = await chrome.tabs.query({
      url: ['*://www.youtube.com/*', '*://youtube.com/*', '*://youtu.be/*']
    });
    
    const tabIds = youTubeTabs.map(tab => tab.id);
    
    if (tabIds.length > 0) {
      await chrome.tabs.remove(tabIds);

      // Clear tracking for closed tabs
      tabIds.forEach(tabId => {
        tabsToCloseAfterVideo.delete(tabId);
        originalFavicons.delete(tabId);
      });
      await persistMarkedTabs();
      await persistOriginalFavicons();

      // Update icon visibility after closing tabs
      await updateIconVisibility();
    }
  } catch (error) {
    console.error('Error closing YouTube tabs:', error);
  }
}


// Handle auto-close videos setting update
async function handleAutoCloseVideosUpdate(enabled) {
  console.log('Auto-close videos setting updated:', enabled);
  await chrome.storage.sync.set({ autoCloseVideos: enabled });
  
  // Update icon state and context menu
  await updateIconState(enabled);
  await updateContextMenu();

  // Reflect favicon behavior to match "marked" when auto-close is enabled
  const youTubeWatchTabs = await chrome.tabs.query({ 
    url: ['*://www.youtube.com/watch*', '*://youtube.com/watch*', '*://youtu.be/*'] 
  });
  
  if (enabled) {
    for (const tab of youTubeWatchTabs) {
      try {
        chrome.tabs.sendMessage(tab.id, { action: 'startVideoEndMonitoring' }).catch(() => {});
      } catch {}
      try {
        // Skip tabs with directional favicons - they override auto-close favicons
        if (directionalFaviconTabs.has(tab.id)) {
          continue;
        }
        await ensureOriginalFaviconCached(tab.id, tab.favIconUrl);
        await chrome.scripting.executeScript({ target: { tabId: tab.id }, func: setCustomFavicon });
      } catch {}
    }
  } else {
    // Restore original favicons for unmarked tabs; keep custom for marked tabs and directional favicons
    for (const tab of youTubeWatchTabs) {
      if (!tabsToCloseAfterVideo.has(tab.id) && !directionalFaviconTabs.has(tab.id)) {
        try {
          await changeTabFavicon(tab.id, false);
        } catch {}
      }
    }
  }
  
  // Reapply directional favicons after auto-close update to ensure they override
  try {
    const { switchToAdjacentTab, switchDirection = 'right' } = await chrome.storage.sync.get(['switchToAdjacentTab', 'switchDirection']);
    if (switchToAdjacentTab) {
      await applyDirectionFavicons(switchDirection, switchToAdjacentTab);
    }
  } catch (e) {
    logDebug('Reapply directional favicons after auto-close update error', e);
  }
}

// Handle switch to adjacent tab setting update
async function handleSwitchToAdjacentTabUpdate(enabled) {
  console.log('Switch to adjacent tab setting updated:', enabled);
  await chrome.storage.sync.set({ switchToAdjacentTab: enabled });
  try {
    const { switchDirection = 'right' } = await chrome.storage.sync.get(['switchDirection']);
    await applyDirectionFavicons(switchDirection || 'right', enabled);
  } catch (e) {
    logDebug('Switch-to-adjacent update direction icon error', e);
  }
}

// Handle switch direction setting update
async function handleSwitchDirectionUpdate(direction) {
  console.log('Switch direction setting updated:', direction);
  await chrome.storage.sync.set({ switchDirection: direction });
  try {
    await applyDirectionFavicons(direction);
  } catch (e) {
    logDebug('Switch direction favicon update error', e);
  }
}

// Handle close delay setting update
function handleCloseDelayUpdate(delay) {
  console.log('Close delay updated:', delay);
  chrome.storage.sync.set({ closeDelay: delay });
}

// Handle click behavior setting update
async function handleClickBehaviorUpdate(clickBehavior) {
  console.log('Click behavior updated:', clickBehavior);
  await chrome.storage.sync.set({ clickBehavior: clickBehavior });
  
  // Update the action behavior based on click preference
  if (clickBehavior === 'popup') {
    // Enable popup for popup mode
    await chrome.action.setPopup({ popup: 'popup.html' });
  } else {
    // Disable popup for toggle modes (autoClose, markTab)
    await chrome.action.setPopup({ popup: '' });
  }
}

// Handle auto-time-action setting updates
function handleAutoTimeActionUpdate(enabled, time = 30, actionType = 'close') {
  console.log('Auto-time-action setting updated:', enabled, 'time:', time, 'action:', actionType);
  chrome.storage.sync.set({ 
    autoTimeAction: enabled,
    autoTimeMinutes: time,
    autoActionType: actionType
  });
  
  if (enabled) {
    if (actionType === 'close') {
      // Start monitoring YouTube tabs for auto-close
      startAutoCloseMonitoring();
      stopAutoPauseMonitoring();
    } else if (actionType === 'pause') {
      // Start monitoring YouTube tabs for auto-pause
      startAutoPauseMonitoring();
      stopAutoCloseMonitoring();
    }
  } else {
    // Stop all monitoring
    stopAutoCloseMonitoring();
    stopAutoPauseMonitoring();
  }
}

// Get YouTube tab count
async function getYouTubeTabCount() {
  try {
    const youTubeTabs = await chrome.tabs.query({
      url: ['*://www.youtube.com/*', '*://youtube.com/*', '*://youtu.be/*']
    });
    return youTubeTabs.length;
  } catch (error) {
    console.error('Error getting YouTube tab count:', error);
    return 0;
  }
}

// Auto-close monitoring (time-based)
let autoCloseInterval = null;

function startAutoCloseMonitoring() {
  // Use alarms in MV3 to wake the service worker reliably
  chrome.alarms.clear('autoCloseTick');
  chrome.alarms.create('autoCloseTick', { periodInMinutes: 5 });
}

function stopAutoCloseMonitoring() {
  chrome.alarms.clear('autoCloseTick');
}

// Auto-pause monitoring
let autoPauseInterval = null;

async function startAutoPauseMonitoring() {
  // Use alarms in MV3 to wake the service worker reliably
  chrome.alarms.clear('autoPauseTick');
  chrome.alarms.create('autoPauseTick', { periodInMinutes: 1 });
}

function stopAutoPauseMonitoring() {
  chrome.alarms.clear('autoPauseTick');
}

const AUTO_CLOSE_ALARM = 'autoCloseTick';
const AUTO_PAUSE_ALARM = 'autoPauseTick';

// Alarms handler and tick bodies
chrome.alarms.onAlarm.addListener(async (alarm) => {
  try {
    if (alarm.name === AUTO_CLOSE_ALARM) {
      await runAutoCloseTick();
    } else if (alarm.name === AUTO_PAUSE_ALARM) {
      await runAutoPauseTick();
    }
  } catch (e) {
    console.error('Error running alarm tick:', e);
  }
});

async function runAutoCloseTick() {
  const settings = await chrome.storage.sync.get(['autoTimeAction', 'autoTimeMinutes', 'autoActionType']);
  if (!settings.autoTimeAction || settings.autoActionType !== 'close') {
    stopAutoCloseMonitoring();
    return;
  }
  const autoCloseTime = (settings.autoTimeMinutes || 30) * 60 * 1000; // ms

  const youTubeTabs = await chrome.tabs.query({
    url: ['*://www.youtube.com/*', '*://youtube.com/*', '*://youtu.be/*']
  });

  for (const tab of youTubeTabs) {
    try {
      const response = await chrome.tabs.sendMessage(tab.id, { action: 'getTimeSpent' });
      if (response && response.timeSpent * 60 * 1000 > autoCloseTime) {
        await forceCloseTab(tab.id);
      }
    } catch (error) {
      logDebug('Could not communicate with tab during autoClose tick:', tab.id, error);
    }
  }
}

async function runAutoPauseTick() {
  const settings = await chrome.storage.sync.get(['autoTimeAction', 'autoTimeMinutes', 'autoActionType']);
  if (!settings.autoTimeAction || settings.autoActionType !== 'pause') {
    stopAutoPauseMonitoring();
    return;
  }
  const autoPauseTime = (settings.autoTimeMinutes || 30) * 60 * 1000; // ms

  const youTubeTabs = await chrome.tabs.query({
    url: ['*://www.youtube.com/*', '*://youtube.com/*', '*://youtu.be/*']
  });

  for (const tab of youTubeTabs) {
    try {
      const response = await chrome.tabs.sendMessage(tab.id, { action: 'getTimeSpent' });
      if (response && response.timeSpent > autoPauseTime / 60000) {
        chrome.tabs.sendMessage(tab.id, { action: 'pauseVideo' });
      }
    } catch (error) {
      logDebug('Could not check time for tab during autoPause tick:', tab.id);
    }
  }
}

// Initialize auto-time-action monitoring based on saved settings
chrome.storage.sync.get(['autoTimeAction', 'autoActionType'], (result) => {
  if (result.autoTimeAction) {
    if (result.autoActionType === 'close') {
      startAutoCloseMonitoring();
    } else if (result.autoActionType === 'pause') {
      startAutoPauseMonitoring();
    }
  }
});

/** React to theme changes in storage to update the pinned icon instantly */
chrome.storage.onChanged.addListener((changes, area) => {
  try {
    if (area === 'sync' && changes.uiTheme) {
      CURRENT_THEME = changes.uiTheme.newValue || 'light';
      chrome.storage.sync.get(['autoCloseVideos'])
        .then(({ autoCloseVideos }) => updateIconState(!!autoCloseVideos))
        .catch(() => {});
    }
  } catch {}
});

// Kick off initial state reconcile when service worker starts
initializeState();
